%get_star_data
% Loads data from Star catalogue.xls spreadsheet and saves to a MATLAB
% structures array for ease of analysis.
% Star data from https://astrosci.scimuze.com/stellar_data.htm
%
% Note Sun parameters:
% Rsun = 696340 km
% Tsun = 5772 K
% AU = 1.496e11 m
% Lsun = 3.846e26 W
%
% And 1Ly = 9.461e15 m and 1 parsec = 3.262 Ly
%
% LAST UPDATED by Andy French Dec 2025

function get_star_data

%Load Excel sheet
[num,txt,raw] = xlsread('Star catalogue.xls','Near Star Cat.');

%How many stars?
dim = size(num); N = dim(1);

%Extract data into a structured array stars
stars.ID = num(:,1);
stars.Hipparcos_ID = num(:,2);
stars.right_ascension = num(:,8);
stars.declination = num(:,9);
stars.distance_parsecs = num(:,10);
stars.apparent_magnitude = num(:,11);
stars.absolute_magnitude = num(:,12);
stars.spectral_type = raw(2:N,13);
stars.colour_index_B_minus_V = num(:,14);

%Calculate star effective temperature (K) using Ballesteros' formula
%https://en.wikipedia.org/wiki/Color_index
BmV = stars.colour_index_B_minus_V;
stars.T_K = 4600*( 1./( 0.92*BmV + 1.7 ) + 1./( 0.92*BmV + 0.62 ) );

%Calculate star luminosity / solar luminosity (Lsun)
%Note Bolometric magnitude Mbol = 4.8 - 2.5*log10( L/Lsun)
stars.L_by_Lsun = 10.^(-stars.absolute_magnitude/2.5) * 10^(4.8/2.5);

%Calculate star radius from Luminosity to radius and temperature
%'black body' relationship
stars.R_by_Rsun = sqrt( stars.L_by_Lsun./ ( (stars.T_K/5772).^4 ) );

%Get max and min color index data
min_B_minus_V = min( stars.colour_index_B_minus_V );
max_B_minus_V = max( stars.colour_index_B_minus_V );

%Set colour index to have max and min values which matches the blue to red
%colour scale
B_minus_V = stars.colour_index_B_minus_V;
B_minus_V( B_minus_V > 1.4) = 1.4;
B_minus_V( B_minus_V < -0.33) = -0.33;

%Get 'jet' colourmap
cmap = colormap('jet'); red = cmap(:,1); green = cmap(:,2); blue = cmap(:,3);
dim = size(cmap); x = linspace( -0.33, 1.4, dim(1) ).';

%Interpolate colourmap based upon colour index to determine [R,G,B] values
%for each star
stars.RGB = [
    interp1(x,red,B_minus_V),...
    interp1(x,green,B_minus_V),...
    interp1(x,blue,B_minus_V)
    ];

%Save data
save star_data stars

%End of code